
//If you're not living on the edge you're taking up too much space

#ifndef _2D_MATH_LIBRARY_H
#define _2D_MATH_LIBRARY_H

#include <math.h>
#include <windows.h>
#include <iostream.h> 


#ifndef _MATH_TOOLS
#define _MATH_TOOLS

#include "trigtables.h"
const float PI			= 3.14159265358979323846264338327950288419716939937510f;
const float PI2			= PI * 2.0f;
const float RADIAN		= PI / 180.0f; 
const float DEGREE360	= 180.0f / PI;

const double RADIAN256 = PI / 128.0f;
const double DEGREE256 = 128.0f / PI;

#define SIN360(theta)	(float)sin((float)theta * RADIAN) 
#define COS360(theta)	(float)cos((float)theta * RADIAN) 
#define TAN360(theta)	(float)tan((float)theta * RADIAN)

#define SIN256(theta)	SinTable[(UCHAR)theta] 
#define COS256(theta)	CosTable[(UCHAR)theta] 
#define TAN256(theta)	TanTable[(UCHAR)theta] 

#define SQRT(a)			((float)sqrt((float)a))
#define SWAP(a,b)		((a)^=(b)^=(a)^=(b))
#define ABS(a)			((a < 0) ? (-(a)) : (a))
#define MIN(a,b)		((a >= b) ? (b) : (a))  
#define MAX(a,b)		((a >= b) ? (a) : (b)) 
#define RAND(a)			(rand() % a)
#endif

inline void Init2DTranslateMat(float mat[3][3], float tx,float ty);
inline void Init2DScaleMat(float mat[3][3], float sx,float sy);
inline void Init2DRotateMat(float mat[3][3], UCHAR theta);

/////////////////////////////////////////////////////////////////////////////////////////
/////CLASS DEFINITIONS///////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
class CVector2D;
class CPosition2D;
class CMatrix1x3;
class CMatrix3x3;
class CVertexStrip2D;
class CPointList2D;
//class CMatrix1x4;
//class CMatrix4x4;


class CVector2D
{
	friend CPosition2D;
public:
	CVector2D()						{ Set(0, 0);	 }
	CVector2D(UCHAR angle)			{ Set(angle);    }
	CVector2D(float xv, float yv)	{ Set(xv, yv);   }

	inline void Clear(void);

	inline void Set(UCHAR angle);
	inline void Set(float xv, float yv);
	inline float GetXV(void);
	inline float GetYV(void);
	inline float Length(void);

	inline void Normalize(void); 

	inline float DotProduct(CVector2D &V);
	inline UCHAR ComputeAngle(CVector2D &V);

	friend float DotProduct(CVector2D &U, CVector2D &V); 
	friend UCHAR ComputeAngle(CVector2D &U, CVector2D &V);

	inline CVector2D& operator =(const CVector2D &other);
	inline CVector2D operator +(const CVector2D &other);
	inline CVector2D operator -(const CVector2D &other);
	inline CVector2D operator *(const float value);
	inline CVector2D operator /(const float value);

	inline CVector2D& operator +=(const CVector2D &other);
	inline CVector2D& operator -=(const CVector2D &other);
	inline CVector2D& operator *=(const float value);
	inline CVector2D& operator /=(const float value);

private:

	float xv, yv;
};


class CPosition2D
{
public:
	CPosition2D()					{ Set(0, 0);	}
	CPosition2D(float x, float y)	{ Set(x, y);	}

	inline void Clear(void);

	inline void Set(float x, float y);
	inline float GetX(void);
	inline float GetY(void);
	inline float Distance(CPosition2D &P);

	friend float Distance(CPosition2D &P1, CPosition2D &P2);

	inline CPosition2D& operator =(const CPosition2D &other);
	inline CPosition2D operator +(const CPosition2D &other);
	inline CPosition2D operator -(const CPosition2D &other);
	inline CPosition2D operator +(const CVector2D &other);
	inline CPosition2D operator -(const CVector2D &other);
	inline CPosition2D& operator +=(const CPosition2D &other);
	inline CPosition2D& operator -=(const CPosition2D &other);
	inline CPosition2D& operator +=(const CVector2D &other);
	inline CPosition2D& operator -=(const CVector2D &other);

private:

	float x, y;
};


class CMatrix1x3
{
public:
	CMatrix1x3()						{ Zero();			}
	CMatrix1x3(CMatrix1x3 &other)		{ *this = other;	}
	CMatrix1x3(const float other[3])	{ *this = other;	}
	
	inline void Zero(void);

	inline void Set(int j, float value);
	inline void Set(float i0j0, float i0j1, float i0j2);
	inline float Get(int j);

	inline CMatrix1x3& operator =(const CMatrix1x3 &other);
	inline CMatrix1x3& operator =(const float other[3]);
	inline CMatrix1x3 operator +(const CMatrix1x3 &other);
	inline CMatrix1x3 operator -(const CMatrix1x3 &other);
	inline CMatrix1x3 operator +(const float other[3]);
	inline CMatrix1x3 operator -(const float other[3]);
	inline CMatrix1x3 operator *(const float value);
	inline CMatrix1x3 operator *(const CMatrix3x3 &other);
	inline CMatrix1x3 operator *(const float other[3][3]);

	inline CMatrix1x3& operator +=(const CMatrix1x3 &other);
	inline CMatrix1x3& operator -=(const CMatrix1x3 &other);
	inline CMatrix1x3& operator +=(const float other[3]);
	inline CMatrix1x3& operator -=(const float other[3]);
	inline CMatrix1x3& operator *=(const float value);
	inline CMatrix1x3& operator *=(const CMatrix3x3 &other);
	inline CMatrix1x3& operator *=(const float other[3][3]);

	void Print(void)
	{
		printf("|  %8.3f  %8.3f  %8.3f  |\n\n", mat1x3[0], mat1x3[1], mat1x3[2]);
	}
private:

	float mat1x3[3];
};


class CMatrix3x3
{
	friend CMatrix1x3;
public:
	CMatrix3x3()						{ Zero();			}
	CMatrix3x3(CMatrix3x3 &other)		{ *this = other;	}
	CMatrix3x3(const float other[3][3])	{ *this = other;	}
	
	inline void Zero(void);
	inline void Identity(void);

	inline void Set(int i, int j, float value);
	inline void Set(float i0j0, float i0j1, float i0j2,
					float i1j0, float i1j1, float i1j2,
					float i2j0, float i2j1, float i2j2);
	inline float Get(int i, int j);

	inline CMatrix3x3& operator =(const CMatrix3x3 &other);
	inline CMatrix3x3& operator =(const float other[3][3]);
	inline CMatrix3x3 operator +(const CMatrix3x3 &other);
	inline CMatrix3x3 operator -(const CMatrix3x3 &other);
	inline CMatrix3x3 operator +(const float other[3][3]);
	inline CMatrix3x3 operator -(const float other[3][3]);
	inline CMatrix3x3 operator *(const float value);
	inline CMatrix3x3 operator *(const CMatrix3x3 &other);
	inline CMatrix3x3 operator *(const float other[3][3]);

	inline CMatrix3x3& operator +=(const CMatrix3x3 &other);
	inline CMatrix3x3& operator -=(const CMatrix3x3 &other);
	inline CMatrix3x3& operator +=(const float other[3][3]);
	inline CMatrix3x3& operator -=(const float other[3][3]);
	inline CMatrix3x3& operator *=(const float value);
	inline CMatrix3x3& operator *=(const CMatrix3x3 &other);
	inline CMatrix3x3& operator *=(const float other[3][3]);

	void Print(void)
	{
		printf("|  %8.3f  %8.3f  %8.3f  |\n", mat3x3[0][0], mat3x3[0][1], mat3x3[0][2]);
		printf("|  %8.3f  %8.3f  %8.3f  |\n", mat3x3[1][0], mat3x3[1][1], mat3x3[1][2]);
		printf("|  %8.3f  %8.3f  %8.3f  |\n\n", mat3x3[2][0], mat3x3[2][1], mat3x3[2][2]);
	}
private:

	float mat3x3[3][3];
};


typedef struct Vertex2D
{
	CMatrix1x3 local;	//local x, y
	CMatrix1x3 aligned; //aligned x, y
	Vertex2D *next;
} Vertex2D;


class CVertexStrip2D
{
public:
	
	CVertexStrip2D();
	CVertexStrip2D(int size);
	CVertexStrip2D(CVertexStrip2D &other);
	~CVertexStrip2D();

	inline void Create(int size);
	inline void Destroy(void);

	inline void SetVertex(int index, float x, float y);
	inline int GetNumVertices(void) { return num_vertices; }
	inline Vertex2D& GetVertex(int i) { return VertexStrip[i]; }
	
	inline void Transform(UCHAR angle, float sx, float sy, float dx, float dy); //M = T * S * R

	inline CVertexStrip2D& operator =(CVertexStrip2D &other);
	inline Vertex2D& operator [](int i) { return VertexStrip[i]; }

private:

	int num_vertices;
	Vertex2D *VertexStrip;
};


class CPointList2D
{
public:
	CPointList2D(int _alloc_increment = 10);
	CPointList2D(CPointList2D &other);
	~CPointList2D();

	inline void Destroy(void);

	inline void AddPoint(int i);
	inline void SetPoint(int index, int i) { PointList[index] = i; }

	inline int GetNumPoints(void) { return num_points; }
	inline int GetPoint(int i) { return PointList[i]; }

	inline CPointList2D& operator =(CPointList2D &other);
	inline int& operator [](int i) { return PointList[i]; }

private:

	int num_points;
	int max_points;
	int alloc_increment;
	int *PointList;
};

/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CLASS DEFINITIONS////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////FUNCTION DEFINITIONS////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
inline void Init2DTranslateMat(float mat[3][3], float dx,float dy)
{
	mat[0][0] = 1;   mat[0][1] = 0;   mat[0][2] = 0; 
	mat[1][0] = 0;   mat[1][1] = 1;   mat[1][2] = 0;
	mat[2][0] = dx;  mat[2][1] = dy;  mat[2][2] = 1;
}

inline void Init2DScaleMat(float mat[3][3], float sx,float sy)
{
	mat[0][0] = sx;  mat[0][1] = 0;   mat[0][2] = 0; 
	mat[1][0] = 0;   mat[1][1] = sy;  mat[1][2] = 0;
	mat[2][0] = 0;   mat[2][1] = 0;   mat[2][2] = 1;
}

inline void Init2DRotateMat(float mat[3][3], UCHAR theta)
{
	mat[0][0] = COS256(theta);    mat[0][1] = SIN256(theta);  mat[0][2] = 0; 
	mat[1][0] = -SIN256(theta);   mat[1][1] = COS256(theta);  mat[1][2] = 0;
	mat[2][0] = 0;                mat[2][1] = 0;              mat[2][2] = 1;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF FUNCTION DEFINITIONS/////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CVECTOR2D CLASS MEMBERS/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
inline void CVector2D::Clear(void)
{ 
	Set(0, 0); 
}

inline void CVector2D::Set(UCHAR angle)			
{ 
	xv = COS256(angle); 
	yv = SIN256(angle); 
}

inline void CVector2D::Set(float xv, float yv)    
{ 
	CVector2D::xv = xv; 
	CVector2D::yv = yv; 
}

inline float CVector2D::GetXV(void)	{ return xv; }
inline float CVector2D::GetYV(void)	{ return yv; }

inline float CVector2D::Length(void)				
{ 
	return SQRT( (xv * xv) + (yv * yv) ); 
}

inline void CVector2D::Normalize(void)
{
	if( (xv == 0) && (yv == 0) ) return;
	*this /= Length();
}

inline float CVector2D::DotProduct(CVector2D &V)	
{ 
	return( (xv * V.xv) + (yv * V.yv) ); 
}

inline UCHAR CVector2D::ComputeAngle(CVector2D &V)
{
	float value;

	value = Length() * V.Length();
	if(value == 0) return 0;

	return (UCHAR)(DEGREE256 * acos(DotProduct(V) / value));
}

inline CVector2D& CVector2D::operator =(const CVector2D &other)		
{ 
	xv = other.xv;
	yv = other.yv; 
	return *this;		  
}
  
inline CVector2D CVector2D::operator +(const CVector2D &other)		
{ 
	return CVector2D(xv + other.xv, yv + other.yv); 
}

inline CVector2D CVector2D::operator -(const CVector2D &other)		
{ 
	return CVector2D(xv - other.xv, yv - other.yv); 
}

inline CVector2D CVector2D::operator *(const float value)				
{ 
	return CVector2D(xv * value, yv * value);		  
}

inline CVector2D CVector2D::operator /(const float value)				
{ 
	return CVector2D(xv / value, yv / value);		  
}

inline CVector2D& CVector2D::operator +=(const CVector2D &other)		
{ 
	xv += other.xv; 
	yv += other.yv; 
	return *this;	  
}

inline CVector2D& CVector2D::operator -=(const CVector2D &other)		
{ 
	xv -= other.xv; 
	yv -= other.yv; 
	return *this;	  
}

inline CVector2D& CVector2D::operator *=(const float value)			
{ 
	xv *= value; 
	yv *= value; 
	return *this;		  
}

inline CVector2D& CVector2D::operator /=(const float value)			
{ 
	xv /= value; 
	yv /= value; 
	return *this;		  
}


float DotProduct(CVector2D &U, CVector2D &V) 
{ 
	return( (U.xv * V.xv) + (U.yv * V.yv) ); 
}


UCHAR ComputeAngle(CVector2D &U, CVector2D &V)
{
	float value;

	value = U.Length() * V.Length();
	if(value == 0) return 0;

	return (UCHAR)(DEGREE256 * acos(DotProduct(U, V) / value));
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CVECTOR2D CLASS MEMBERS//////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
	

/////////////////////////////////////////////////////////////////////////////////////////
/////CPOSITION2D CLASS MEMBERS///////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
inline void CPosition2D::Clear(void)
{
	Set(0, 0);
}

inline void CPosition2D::Set(float x, float y)
{
	CPosition2D::x = x;
	CPosition2D::y = y;
}

inline float CPosition2D::GetX(void) { return x; }
inline float CPosition2D::GetY(void) { return y; }

inline float CPosition2D::Distance(CPosition2D &P)
{
	float xd = x - P.x,
		  yd = y - P.y;

	return SQRT( (xd * xd) + (yd * yd) );
}

inline CPosition2D& CPosition2D::operator =(const CPosition2D &other)
{
	x = other.x;
	y = other.y;
	return *this;
}

inline CPosition2D CPosition2D::operator +(const CPosition2D &other)
{
	return CPosition2D(x + other.x, y + other.y);
}

inline CPosition2D CPosition2D::operator -(const CPosition2D &other)
{
	return CPosition2D(x - other.x, y - other.y);
}

inline CPosition2D CPosition2D::operator +(const CVector2D &other)
{
	return CPosition2D(x + other.xv, y + other.yv);
}

inline CPosition2D CPosition2D::operator -(const CVector2D &other)
{
	return CPosition2D(x - other.xv, y - other.yv);
}

inline CPosition2D& CPosition2D::operator +=(const CPosition2D &other)
{
	x += other.x;
	y += other.y;
	return *this;
}

inline CPosition2D& CPosition2D::operator -=(const CPosition2D &other)
{
	x -= other.x;
	y -= other.y;
	return *this;	
}

inline CPosition2D& CPosition2D::operator +=(const CVector2D &other)
{
	x += other.xv;
	y += other.yv;
	return *this;
}

inline CPosition2D& CPosition2D::operator -=(const CVector2D &other)
{
	x -= other.xv;
	y -= other.yv;
	return *this;
}

float Distance(CPosition2D &P1, CPosition2D &P2)
{
	float xd = P2.x - P1.x,
		  yd = P2.y - P1.y;

	return SQRT( (xd * xd) + (yd * yd) );
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CPOSITION2D CLASS MEMBERS////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CMATRIX1X3 CLASS MEMBERS////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
inline void CMatrix1x3::Zero(void)
{
	mat1x3[0] = 0;	mat1x3[1] = 0;	mat1x3[2] = 0; 
}

inline void CMatrix1x3::Set(int j, float value)
{
	mat1x3[j] = value;
}

inline void CMatrix1x3::Set(float i0j0, float i0j1, float i0j2)
{
	mat1x3[0] = i0j0;  mat1x3[1] = i0j1;  mat1x3[2] = i0j2;
}

inline float CMatrix1x3::Get(int j)
{
	return mat1x3[j];
}

inline CMatrix1x3& CMatrix1x3::operator =(const CMatrix1x3 &other)
{
	mat1x3[0] = other.mat1x3[0];  mat1x3[1] = other.mat1x3[1];  mat1x3[2] = other.mat1x3[2];
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator =(const float other[3])
{
	mat1x3[0] = other[0];  mat1x3[1] = other[1];  mat1x3[2] = other[2];
	return *this;
}

inline CMatrix1x3 CMatrix1x3::operator +(const CMatrix1x3 &other)
{
	CMatrix1x3 mat;
	mat.Set(mat1x3[0] + other.mat1x3[0], mat1x3[1] + other.mat1x3[1], mat1x3[2] + other.mat1x3[2]);
	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator -(const CMatrix1x3 &other)
{
	CMatrix1x3 mat;
	mat.Set(mat1x3[0] - other.mat1x3[0], mat1x3[1] - other.mat1x3[1], mat1x3[2] - other.mat1x3[2]);
	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator +(const float other[3])
{
	CMatrix1x3 mat;
	mat.Set(mat1x3[0] + other[0], mat1x3[1] + other[1], mat1x3[2] + other[2]);
	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator -(const float other[3])
{
	CMatrix1x3 mat;
	mat.Set(mat1x3[0] - other[0], mat1x3[1] - other[1], mat1x3[2] - other[2]);
	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator *(const float value)
{
	CMatrix1x3 mat;
	mat.Set(mat1x3[0] * value, mat1x3[1] * value, mat1x3[2] * value);
	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator *(const CMatrix3x3 &other)
{
	int j;
	CMatrix1x3 mat;
	
	for(j=0; j<3; j++)
	{
		mat.mat1x3[j] = mat1x3[0] * other.mat3x3[0][j] + 
						mat1x3[1] * other.mat3x3[1][j] + 
						mat1x3[2] * other.mat3x3[2][j];
	}

	return mat;
}

inline CMatrix1x3 CMatrix1x3::operator *(const float other[3][3])
{
	int j;
	CMatrix1x3 mat;
	
	for(j=0; j<3; j++)
	{
		mat.mat1x3[j] = mat1x3[0] * other[0][j] + 
						mat1x3[1] * other[1][j] + 
						mat1x3[2] * other[2][j];
	}

	return mat;
}

inline CMatrix1x3& CMatrix1x3::operator +=(const CMatrix1x3 &other)
{
	mat1x3[0] += other.mat1x3[0];  mat1x3[1] += other.mat1x3[1];  mat1x3[2] += other.mat1x3[2];
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator -=(const CMatrix1x3 &other)
{
	mat1x3[0] -= other.mat1x3[0];  mat1x3[1] -= other.mat1x3[1];  mat1x3[2] -= other.mat1x3[2];
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator +=(const float other[3])
{
	mat1x3[0] += other[0];  mat1x3[1] += other[1];  mat1x3[2] += other[2];
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator -=(const float other[3])
{
	mat1x3[0] -= other[0];  mat1x3[1] -= other[1];  mat1x3[2] -= other[2];
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator *=(const float value)
{
	mat1x3[0] *= value;  mat1x3[1] *= value;  mat1x3[2] *= value;
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator *=(const CMatrix3x3 &other)
{
	int j;
	CMatrix1x3 mat;
	
	for(j=0; j<3; j++)
	{
		mat.mat1x3[j] = mat1x3[0] * other.mat3x3[0][j] + 
						mat1x3[1] * other.mat3x3[1][j] + 
						mat1x3[2] * other.mat3x3[2][j];
	}

	*this = mat;
	return *this;
}

inline CMatrix1x3& CMatrix1x3::operator *=(const float other[3][3])
{
	int j;
	CMatrix1x3 mat;
	
	for(j=0; j<3; j++)
	{
		mat.mat1x3[j] = mat1x3[0] * other[0][j] + 
						mat1x3[1] * other[1][j] + 
						mat1x3[2] * other[2][j];
	}

	*this = mat;
	return *this;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CMATRIX1X3 CLASS MEMBERS/////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CMATRIX3X3 CLASS MEMBERS////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
inline void CMatrix3x3::Zero(void)
{
	mat3x3[0][0] = 0;  mat3x3[0][1] = 0;  mat3x3[0][2] = 0;
	mat3x3[1][0] = 0;  mat3x3[1][1] = 0;  mat3x3[1][2] = 0;
	mat3x3[2][0] = 0;  mat3x3[2][1] = 0;  mat3x3[2][2] = 0;
}

inline void CMatrix3x3::Identity(void)
{
	mat3x3[0][0] = 1;  mat3x3[0][1] = 0;  mat3x3[0][2] = 0;
	mat3x3[1][0] = 0;  mat3x3[1][1] = 1;  mat3x3[1][2] = 0;
	mat3x3[2][0] = 0;  mat3x3[2][1] = 0;  mat3x3[2][2] = 1;
}

inline void CMatrix3x3::Set(int i, int j, float value)
{
	mat3x3[i][j] = value;
}

inline void CMatrix3x3::Set(float i0j0, float i0j1, float i0j2,
							float i1j0, float i1j1, float i1j2,
							float i2j0, float i2j1, float i2j2)
{
	mat3x3[0][0] = i0j0;  mat3x3[0][1] = i0j1;  mat3x3[0][2] = i0j2;
	mat3x3[1][0] = i1j0;  mat3x3[1][1] = i1j1;  mat3x3[1][2] = i1j2;
	mat3x3[2][0] = i2j0;  mat3x3[2][1] = i2j1;  mat3x3[2][2] = i2j2;
}

inline float CMatrix3x3::Get(int i, int j)
{
	return mat3x3[i][j];
}

inline CMatrix3x3& CMatrix3x3::operator =(const CMatrix3x3 &other)
{
	mat3x3[0][0] = other.mat3x3[0][0];  mat3x3[0][1] = other.mat3x3[0][1];  mat3x3[0][2] = other.mat3x3[0][2];
	mat3x3[1][0] = other.mat3x3[1][0];  mat3x3[1][1] = other.mat3x3[1][1];  mat3x3[1][2] = other.mat3x3[1][2];
	mat3x3[2][0] = other.mat3x3[2][0];  mat3x3[2][1] = other.mat3x3[2][1];  mat3x3[2][2] = other.mat3x3[2][2];
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator =(const float other[3][3])
{
	mat3x3[0][0] = other[0][0];  mat3x3[0][1] = other[0][1];  mat3x3[0][2] = other[0][2];
	mat3x3[1][0] = other[1][0];  mat3x3[1][1] = other[1][1];  mat3x3[1][2] = other[1][2];
	mat3x3[2][0] = other[2][0];  mat3x3[2][1] = other[2][1];  mat3x3[2][2] = other[2][2];
	return *this;
}

inline CMatrix3x3 CMatrix3x3::operator +(const CMatrix3x3 &other)
{
	CMatrix3x3 mat;
	mat.Set(mat3x3[0][0] + other.mat3x3[0][0], mat3x3[0][1] + other.mat3x3[0][1], mat3x3[0][2] + other.mat3x3[0][2], 
			mat3x3[1][0] + other.mat3x3[1][0], mat3x3[1][1] + other.mat3x3[1][1], mat3x3[1][2] + other.mat3x3[1][2],
			mat3x3[2][0] + other.mat3x3[2][0], mat3x3[2][1] + other.mat3x3[2][1], mat3x3[2][2] + other.mat3x3[2][2]);
	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator -(const CMatrix3x3 &other)
{
	CMatrix3x3 mat;
	mat.Set(mat3x3[0][0] - other.mat3x3[0][0], mat3x3[0][1] - other.mat3x3[0][1], mat3x3[0][2] - other.mat3x3[0][2], 
			mat3x3[1][0] - other.mat3x3[1][0], mat3x3[1][1] - other.mat3x3[1][1], mat3x3[1][2] - other.mat3x3[1][2],
			mat3x3[2][0] - other.mat3x3[2][0], mat3x3[2][1] - other.mat3x3[2][1], mat3x3[2][2] - other.mat3x3[2][2]);
	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator +(const float other[3][3])
{
	CMatrix3x3 mat;
	mat.Set(mat3x3[0][0] + other[0][0], mat3x3[0][1] + other[0][1], mat3x3[0][2] + other[0][2], 
			mat3x3[1][0] + other[1][0], mat3x3[1][1] + other[1][1], mat3x3[1][2] + other[1][2],
			mat3x3[2][0] + other[2][0], mat3x3[2][1] + other[2][1], mat3x3[2][2] + other[2][2]);
	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator -(const float other[3][3])
{
	CMatrix3x3 mat;
	mat.Set(mat3x3[0][0] - other[0][0], mat3x3[0][1] - other[0][1], mat3x3[0][2] - other[0][2], 
			mat3x3[1][0] - other[1][0], mat3x3[1][1] - other[1][1], mat3x3[1][2] - other[1][2],
			mat3x3[2][0] - other[2][0], mat3x3[2][1] - other[2][1], mat3x3[2][2] - other[2][2]);
	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator *(const float value)
{
	CMatrix3x3 mat;
	mat.Set(mat3x3[0][0] * value, mat3x3[0][1] * value, mat3x3[0][2] * value, 
			mat3x3[1][0] * value, mat3x3[1][1] * value, mat3x3[1][2] * value,
			mat3x3[2][0] * value, mat3x3[2][1] * value, mat3x3[2][2] * value);
	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator *(const CMatrix3x3 &other)
{
	int i, j;
	CMatrix3x3 mat;
	
	for(i=0; i<3; i++)
	{
		for(j=0; j<3; j++)
		{
			mat.mat3x3[i][j] = mat3x3[i][0] * other.mat3x3[0][j] + 
							   mat3x3[i][1] * other.mat3x3[1][j] + 
							   mat3x3[i][2] * other.mat3x3[2][j];
		}
	}

	return mat;
}

inline CMatrix3x3 CMatrix3x3::operator *(const float other[3][3])
{
	int i, j;
	CMatrix3x3 mat;
	
	for(i=0; i<3; i++)
	{
		for(j=0; j<3; j++)
		{
			mat.mat3x3[i][j] = mat3x3[i][0] * other[0][j] + 
							   mat3x3[i][1] * other[1][j] + 
							   mat3x3[i][2] * other[2][j];
		}
	}

	return mat;
}

inline CMatrix3x3& CMatrix3x3::operator +=(const CMatrix3x3 &other)
{
	mat3x3[0][0] += other.mat3x3[0][0];  mat3x3[0][1] += other.mat3x3[0][1];  mat3x3[0][2] += other.mat3x3[0][2];
	mat3x3[1][0] += other.mat3x3[1][0];  mat3x3[1][1] += other.mat3x3[1][1];  mat3x3[1][2] += other.mat3x3[1][2];
	mat3x3[2][0] += other.mat3x3[2][0];  mat3x3[2][1] += other.mat3x3[2][1];  mat3x3[2][2] += other.mat3x3[2][2];
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator -=(const CMatrix3x3 &other)
{
	mat3x3[0][0] -= other.mat3x3[0][0];  mat3x3[0][1] -= other.mat3x3[0][1];  mat3x3[0][2] -= other.mat3x3[0][2];
	mat3x3[1][0] -= other.mat3x3[1][0];  mat3x3[1][1] -= other.mat3x3[1][1];  mat3x3[1][2] -= other.mat3x3[1][2];
	mat3x3[2][0] -= other.mat3x3[2][0];  mat3x3[2][1] -= other.mat3x3[2][1];  mat3x3[2][2] -= other.mat3x3[2][2];
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator +=(const float other[3][3])
{
	mat3x3[0][0] += other[0][0];  mat3x3[0][1] += other[0][1];  mat3x3[0][2] += other[0][2];
	mat3x3[1][0] += other[1][0];  mat3x3[1][1] += other[1][1];  mat3x3[1][2] += other[1][2];
	mat3x3[2][0] += other[2][0];  mat3x3[2][1] += other[2][1];  mat3x3[2][2] += other[2][2];
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator -=(const float other[3][3])
{
	mat3x3[0][0] -= other[0][0];  mat3x3[0][1] -= other[0][1];  mat3x3[0][2] -= other[0][2];
	mat3x3[1][0] -= other[1][0];  mat3x3[1][1] -= other[1][1];  mat3x3[1][2] -= other[1][2];
	mat3x3[2][0] -= other[2][0];  mat3x3[2][1] -= other[2][1];  mat3x3[2][2] -= other[2][2];
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator *=(const float value)
{
	mat3x3[0][0] *= value;  mat3x3[0][1] *= value;  mat3x3[0][2] *= value; 
	mat3x3[1][0] *= value;  mat3x3[1][1] *= value;  mat3x3[1][2] *= value;
	mat3x3[2][0] *= value;  mat3x3[2][1] *= value;  mat3x3[2][2] *= value;
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator *=(const CMatrix3x3 &other)
{
	int i, j;
	CMatrix3x3 mat;
	
	for(i=0; i<3; i++)
	{
		for(j=0; j<3; j++)
		{
			mat.mat3x3[i][j] = mat3x3[i][0] * other.mat3x3[0][j] + 
							   mat3x3[i][1] * other.mat3x3[1][j] + 
							   mat3x3[i][2] * other.mat3x3[2][j];
		}
	}

	*this = mat;
	return *this;
}

inline CMatrix3x3& CMatrix3x3::operator *=(const float other[3][3])
{
	int i, j;
	CMatrix3x3 mat;
	
	for(i=0; i<3; i++)
	{
		for(j=0; j<3; j++)
		{
			mat.mat3x3[i][j] = mat3x3[i][0] * other[0][j] + 
							   mat3x3[i][1] * other[1][j] + 
							   mat3x3[i][2] * other[2][j];
		}
	}

	*this = mat;
	return *this;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CMATRIX3X3 CLASS MEMBERS/////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CVERTEXSTRIP2D CLASS MEMBERS////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

CVertexStrip2D::CVertexStrip2D()
{
	if(_alloc_increment < 1) _alloc_increment = 1;
	num_vertices = 0;
	max_vertices = 0;
	alloc_increment = _alloc_increment;
	curr_ptr = NULL;
	VertexStrip = NULL;
}

CVertexStrip2D::CVertexStrip2D(CVertexStrip2D &other)
{
	*this = other;
}

CVertexStrip2D::~CVertexStrip2D()
{
	Destroy();
}

inline void CVertexStrip2D::Destroy(void)
{
	Vertex2D *tmp;
	num_vertices = 0;
	
	if(VertexStrip != NULL)
	{
		curr_ptr = VertexStrip;
		while(curr_ptr->next != NULL)
		{
			tmp = curr_ptr;
			curr_ptr = curr_ptr->next;
			delete tmp;
			tmp = NULL;
		}
	}
	VertexStrip = NULL;
	curr_ptr = NULL;
}

inline void CVertexStrip2D::AddVertex(float x, float y)
{
	if(VertexStrip == NULL)
	{
		num_vertices = 0;
		VertexStrip = new Vertex2D;
		curr_ptr = VertexStrip;
		curr_ptr->next = NULL;
	}

	VertexStrip[num_vertices].local.Set(x, y, 1);
	num_vertices++;

	if(num_vertices >= max_vertices)
	{
		max_vertices += alloc_increment;
		Vertex2D *tmp_strip = new Vertex2D[max_vertices];
		for(int i=0; i<num_vertices; i++)
		{
			tmp_strip[i].local = VertexStrip[i].local;
		}
		delete[] VertexStrip;
		VertexStrip = tmp_strip;
		tmp_strip = NULL;
	}
}

inline void CVertexStrip2D::Transform(UCHAR angle, float sx, float sy, float dx, float dy)
{
	CMatrix3x3 M, T, S, R;

	T.Set(  1,  0,  0,
		    0,  1,  0,
		    dx, dy, 1  );

	S.Set(  sx,  0,  0,
		    0,   sy, 0,
			0,   0,  1  );

	R.Set(  COS256(angle),  SIN256(angle),  0,
		   -SIN256(angle),  COS256(angle),  0,
		    0,              0,              1  );

	M = S * R * T;//T * S * R;

	for(int i=0; i<num_vertices; i++)
	{
		VertexStrip[i].aligned = VertexStrip[i].local * M;
	}
}

inline CVertexStrip2D& CVertexStrip2D::operator =(CVertexStrip2D &other)
{
	int i;
	num_vertices = other.num_vertices;
	max_vertices = other.max_vertices;
	alloc_increment = other.alloc_increment;
	
	if(VertexStrip != NULL) 
	{
		delete[] VertexStrip;
		VertexStrip = NULL;
	}
	
	VertexStrip = new Vertex2D[max_vertices];
	
	for(i=0; i<num_vertices; i++)
	{
		VertexStrip[i].local = other.VertexStrip[i].local;
		VertexStrip[i].aligned = other.VertexStrip[i].aligned;
	}

	return *this;
}
/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CVERTEXSTRIP2D CLASS MEMBERS/////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////
/////CPOINTLIST2D CLASS MEMBERS//////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
CPointList2D::CPointList2D(int _alloc_increment)
{
	if(_alloc_increment < 1) _alloc_increment = 1;
	num_points = 0;
	max_points = 0;
	alloc_increment = _alloc_increment;
	PointList = NULL;
}

CPointList2D::CPointList2D(CPointList2D &other)
{
	*this = other;
}

CPointList2D::~CPointList2D()
{
	Destroy();
	alloc_increment = 0;
}

inline void CPointList2D::Destroy(void)
{
	num_points = 0;
	max_points = 0;
	
	if(PointList != NULL)
	{
		delete[] PointList;
		PointList = NULL;
	}
}

inline void CPointList2D::AddPoint(int i)
{
	if(PointList == NULL)
	{
		num_points = 0;
		max_points = alloc_increment;
		PointList = new int[max_points];
	}

	PointList[num_points] = i;
	num_points++;

	if(num_points >= max_points)
	{
		max_points += alloc_increment;
		int *tmp_list = new int[max_points];
		for(int i=0; i<num_points; i++)
		{
			tmp_list[i] = PointList[i];
		}
		delete[] PointList;
		PointList = tmp_list;
		tmp_list = NULL;
	}
}

inline CPointList2D& CPointList2D::operator =(CPointList2D &other)
{
	int i;
	num_points = other.num_points;
	max_points = other.max_points;
	alloc_increment = other.alloc_increment;
	
	if(PointList != NULL) 
	{
		delete[] PointList;
		PointList = NULL;
	}
	
	PointList = new int[max_points];
	
	for(i=0; i<num_points; i++)
	{
		PointList[i] = other.PointList[i];
	}

	return *this;
}

/////////////////////////////////////////////////////////////////////////////////////////
/////END OF CPOINTLIST2D CLASS MEMBERS///////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
#endif